/* smfplay.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>
#include <sys/types.h>
#ifndef NOSETPRIO
#include <sys/rtprio.h>
#endif

extern void
attach_midi_port(void);
extern void
reset_all_midi(void);
extern void
put_midi_message(int c);

typedef unsigned char byte_t;
typedef byte_t *buffer_t;
typedef byte_t const *cbuffer_t;
typedef unsigned short word_t;
typedef unsigned long dword_t;
#ifndef Bool
#define Bool int
#define True (!0)
#define False (0)
#endif

#define MAGIC_HEADER_CHUNK     "MThd"
#define LEN_MAGIC_HEADER_CHUNK 4
#define MAGIC_TRACK_CHUNK      "MTrk"
#define LEN_MAGIC_TRACK_CHUNK  4
#define MIN_HEADER_LENGTH 6
#define INITIAL_BUFFER_SIZE 256

/* start margin in usec */
#define START_MARGIN_TIME (200000)

/* event structure */
typedef struct
{
    byte_t eventCode;
    int lenParameter;
    cbuffer_t cRefParameters;
} event_t;    

/* header chunk info */
typedef struct
{
    dword_t length;
    int format;
    int numTracks;
    unsigned int timeUnitValue;
    enum {
        ETUQPM,
        ETUHMSF
    } timeUnitType;
} header_chunk_t;

/* track info */
typedef struct
{
    dword_t length;
    dword_t remain;
    byte_t prevEvent;
    byte_t portNumber;
    cbuffer_t buffer;
} track_info_t;


/* tick generator */
typedef struct
{
    long tempo;    /* microseconds for an unit of timebase */
    dword_t curTick;
    struct timeval lastGetTimeOfDay;
} tick_t;

/* SMF pool buffer */
typedef struct
{
    buffer_t pool;
    dword_t lenPool;
    track_info_t *tracks;
    tick_t tick;
    header_chunk_t header;
    Bool flQuiet;
    byte_t portNumber;
} smf_buffer_t;


/* event priority queue */
typedef struct tag_event_queue_node_t
{
    dword_t ariseTick;
    event_t event;
    track_info_t *refEventOwner;
    struct tag_event_queue_node_t *toLower;
    /* link to high priority (ariseTick is smaller)
       from low priority (ariseTick is larger) */
} event_queue_node_t;

typedef struct
{
    event_queue_node_t *topPriority; /* highest priority */
} event_queue_t;


/**************************************************
  kinds of get variable
 */

/* get variable length value */
dword_t
get_var_length_value(cbuffer_t *refBuf, int *refCount)
{
    dword_t result = 0;
    byte_t c;

    if (NULL != refCount) {
        *refCount = 0;
    }

    do {
        if (NULL != refCount) {
            ++*refCount;
        }
        c = **refBuf;
        result <<= 7;
        result |= c & 0x7f;
        ++*refBuf;
    } while (0 != (c & 0x80));

    return result;
}


/* get fix length byte */
byte_t
get_byte(cbuffer_t *refBuf)
{
    byte_t result;

    result = **refBuf;
    ++*refBuf;
    
    return result;
}

/* get fix length word */
word_t
get_word(cbuffer_t *refBuf)
{
    word_t result;

    result   = **refBuf;
    ++*refBuf;
    result <<= 8;
    result  |= **refBuf;
    ++*refBuf;
    
    return result;
}

/* get fix length dword */
dword_t
get_dword(cbuffer_t *refBuf)
{
    dword_t result;

    result   = **refBuf;
    ++*refBuf;
    result <<= 8;
    result  |= **refBuf;
    ++*refBuf;
    result <<= 8;
    result  |= **refBuf;
    ++*refBuf;
    result <<= 8;
    result  |= **refBuf;
    ++*refBuf;
    
    return result;
}


/*************************************************************************
  list based priority queue
 */


/* constructor */
void
queue_construct(event_queue_t *refQueue /* object */)
{
    refQueue->topPriority = NULL;
}

/* destructor */
void
queue_destruct(event_queue_t *refQueue /* object */)
{
    event_queue_node_t *p, *pn;
    
    for (p=refQueue->topPriority; NULL != p; p=pn) {
        pn = p->toLower;
        free(p);
    }

    refQueue->topPriority = NULL;
}

/* unshift */
void
queue_unshift(event_queue_t *refQueue /* object */,
              dword_t ariseTick, event_t const *refEvent,
              track_info_t *refEventOwner)
{
    event_queue_node_t *p;
    event_queue_node_t *newNode;

    {
        /* treat new node */
        newNode = (event_queue_node_t *)malloc(sizeof(event_queue_node_t));
        newNode->ariseTick = ariseTick;
        newNode->event = *refEvent;
        newNode->refEventOwner = refEventOwner;
    }

    {
        /* search and insert */
        event_queue_node_t **refPrev = &refQueue->topPriority;
        for   (p=refQueue->topPriority; NULL!=p; p=p->toLower) {
            if   (p->ariseTick > ariseTick) {
                newNode->toLower = p;
                *refPrev = newNode;
                goto quit;
            }
            refPrev = &p->toLower;
        }
        /* new event is lowest priority */
        newNode->toLower = NULL;
        *refPrev = newNode;
    }
    
quit:
    return;
}


/* peek top arise tick */
Bool /* isNothing */
queue_peek_top_arise_tick(event_queue_t const *refQueue /* object */,
                          dword_t *refAriseTick /* result */)
{
    Bool result = False;

    if   (NULL != refQueue->topPriority) {
        if (NULL != refAriseTick)
            *refAriseTick = refQueue->topPriority->ariseTick;
    } else
        result = True;

    return result;
}


/* shift */
Bool /* isNothing */
queue_shift(event_queue_t *refQueue /* object */,
            event_t *refEvent /* result */,
            track_info_t **refRefEventOwner /* result */)
{
    Bool result = False;

    if   (NULL != refQueue->topPriority) {
        if (NULL != refEvent)
            *refEvent = refQueue->topPriority->event;
        if (NULL != refRefEventOwner)
            *refRefEventOwner = refQueue->topPriority->refEventOwner;
        refQueue->topPriority = refQueue->topPriority->toLower;
    } else
        result = True;

    return result;
}


/*************************************************
  get kinds of chunk
 */


/* get header chunk */
Bool /* result isIllegal */
get_header_chunk(cbuffer_t *refBuf, header_chunk_t *refHeader)
{
    Bool result = False;
    dword_t lenHeader;

    {
        /* check magic number */
        if (memcmp(*refBuf, MAGIC_HEADER_CHUNK, LEN_MAGIC_HEADER_CHUNK)) {
            result = True;
            goto quit;
        }
        *refBuf += LEN_MAGIC_HEADER_CHUNK;
    }
    {
        /* get and check header length */
        refHeader->length = lenHeader = get_dword(refBuf);
        if ( MIN_HEADER_LENGTH > lenHeader ){
            result = True;
            goto quit;
        }
    }
    {
        /* get format */
        refHeader->format = get_word(refBuf);
        lenHeader -= sizeof(word_t);
    }
    {
        /* get number of tracks */
        refHeader->numTracks = get_word(refBuf);
        lenHeader -= sizeof(word_t);
    }
    {
        /* get time unit */
        word_t tmp;
        
        tmp = get_word(refBuf);
        refHeader->timeUnitValue = tmp & 0x7fff;
        if (tmp & 0x8000)
            refHeader->timeUnitType = ETUHMSF;
        else
            refHeader->timeUnitType = ETUQPM;
        lenHeader -= sizeof(word_t);
    }
    *refBuf -= lenHeader;

quit:
    return result;
}


/* get track chunk */
Bool /* result isIllegal */
get_track_chunk(cbuffer_t *refBuf,
                track_info_t *refTrack /* result */)
{
    Bool result = False;

    {
        /* check magic number */
        if (memcmp(*refBuf, MAGIC_TRACK_CHUNK, LEN_MAGIC_TRACK_CHUNK)) {
            result = True;
            goto quit;
        }
        *refBuf += LEN_MAGIC_TRACK_CHUNK;
    }
    {
        /* get length */
        refTrack->length = refTrack->remain = get_dword(refBuf);
    }

quit:
    return result;
}


/************************************************************
  get event
 */

/* get a event */
Bool /* isIllegal */
get_event(track_info_t *refTrack /* I/O */,
          event_t *refEvent /* result */)
{
    Bool result = False;
    cbuffer_t saveTop;
    cbuffer_t *refBuf = &refTrack->buffer;
    byte_t eventTop;

    if (0 == refTrack->remain) {
        result = True;
        refTrack->remain = 0;
        goto quit;
    }

    saveTop = *refBuf;
    eventTop = get_byte(refBuf);
    refTrack->remain--;

    if (0 == (eventTop & 0x80)) {
        /* Running Status used */
        
        eventTop = refTrack->prevEvent;
        --*refBuf; /* put back one byte */
    }

    refEvent->eventCode = eventTop;

    switch (eventTop & 0xf0) {
        /* Regular MIDI event */
    case 0x80:
    case 0x90:
    case 0xa0:
    case 0xb0:
    case 0xe0:
        refEvent->lenParameter = 2;
        refTrack->prevEvent = eventTop;
        break;
    case 0xc0:
    case 0xd0:
        refEvent->lenParameter = 1;
        refTrack->prevEvent = eventTop;
        break;
    case 0xf0:
        switch (eventTop & 0xf) {
        case 0x0:
            /* send exclusive (normal) */
            {
                int count;

                refEvent->lenParameter =
                    get_var_length_value(refBuf, &count);
                if (refTrack->remain < count) {
                    result = True;
                    refTrack->remain = 0;
                    goto quit;
                }
            }
            break;
        case 0x7:
            /* send exclusive (abnormal) */
            {
                int count;

                refEvent->lenParameter =
                    get_var_length_value(refBuf, &count) -1;
                if (refTrack->remain < count) {
                    result = True;
                    refTrack->remain = 0;
                    goto quit;
                }
            }
            eventTop = get_byte(refBuf);
            break;
        case 0x1:
        case 0x3:
        case 0x5:
            refEvent->lenParameter = 1;
            break;
        case 0x2:
            refEvent->lenParameter = 2;
            break;
        case 0x4:
        case 0x9:
        case 0xd:
            fprintf(stderr, "illegal MIDI event 0x%02x\n", eventTop);
            result = True;
            refTrack->remain = 1;
            goto quit;
        case 0x6:
        case 0x8:
        case 0xa:
        case 0xb:
        case 0xc:
        case 0xe:
            refEvent->lenParameter = 0;
            break;
        case 0xf:
            {
                cbuffer_t tmp = *refBuf;
                int count;
                int num;
                if (1 >= refTrack->remain) {
                    result = True;
                    refTrack->remain = 0;
                    goto quit;
                }
                num = get_byte(refBuf);
                refEvent->lenParameter = get_var_length_value(refBuf, &count);
                refEvent->lenParameter += count+1;
                *refBuf = tmp;
                if (num ==0x2f) {
                    /* EOT */
                    refEvent->cRefParameters = *refBuf;
                    *refBuf += refEvent->lenParameter;
                    refTrack->remain = 0;
                }
            }
            break;
        }
    }
    if (refEvent->lenParameter > refTrack->remain) {
        result = True;
        refTrack->remain = 0;
        goto quit;
    }
    refEvent->cRefParameters = *refBuf;
    *refBuf += refEvent->lenParameter;
    refTrack->remain -= refEvent->lenParameter;
    
quit:
    return result;
}


/****************************************************************
  get valid event from track
 */

Bool /* isEnd */
get_valid_event(smf_buffer_t *refSMFBuffer /* changed */,
                track_info_t *refTrack /* changed */,
                event_t *refEvent /* result */,
                dword_t *refDeltaTime /* result */)
{
    Bool result = False;
    
    *refDeltaTime = 0;
    
retry:
    if   (0 == refTrack->remain) {
        result = True;
        goto quit;
    }
    {
        int len;
        *refDeltaTime += get_var_length_value(&refTrack->buffer, &len);
        if   (refTrack->remain <= len) {
            result = True;
            goto quit;
        }
        refTrack->remain -= len;
    }

    if   (get_event(refTrack, refEvent)) {
        result = True;
        goto quit;
    }

    /* modify */
    switch (refEvent->eventCode & 0xf0) {
    case 0xf0:
        switch (refEvent->eventCode & 0x0f) {
        case 0xe:
            goto retry;
        case 0xf:
            switch (*refEvent->cRefParameters) {
            case 0x01:
            case 0x02:
            case 0x03:
                if (!refSMFBuffer->flQuiet) {
                    cbuffer_t p = &refEvent->cRefParameters[1];
                    do {
                        p++;
                    } while (*p & 0x80);
                    fprintf(stderr, "%.*s\n", refEvent->lenParameter-2, p);
                }
                goto retry;
            case 0x21:
                break;
            case 0x51:
                break;
            default:
                goto retry;
            }
        }
    }
quit:
    return result;
}

/**************************************************************
  timer
 */

/* normalize timeval */

static void
normalize_timeval(struct timeval *refTimeval /* I/O */)
{
    while (refTimeval->tv_usec >= 1000000) {
        refTimeval->tv_sec++;
        refTimeval->tv_usec -= 1000000;
    }
    while (refTimeval->tv_usec < 0) {
        refTimeval->tv_sec--;
        refTimeval->tv_usec += 1000000;
    }
}


/* init tick */
void
tick_init(tick_t *refTick /* object */,
          unsigned int timebase, dword_t initialTick)
{
    /*
      120 qpm
      ---  1000000*60/(timebase/120) = 500000/timebase
     */
    refTick->tempo = 500000U/timebase;
    refTick->curTick = initialTick;
}

/* start tick */
void
tick_start(tick_t *refTick /* object */)
{
    gettimeofday(&refTick->lastGetTimeOfDay, NULL);
    {
        /* take margin */
        struct timeval tmp;
        tmp.tv_sec = 0;
        tmp.tv_usec = START_MARGIN_TIME;
        normalize_timeval(&tmp);
        timeradd(&tmp, &refTick->lastGetTimeOfDay, &refTick->lastGetTimeOfDay);
    }
}

/* update tick */
dword_t /* current tick */
tick_update(tick_t *refTick /* object */)
{
    struct timeval curval, diffval;

    gettimeofday(&curval, NULL);
    timersub(&curval, &refTick->lastGetTimeOfDay, &diffval);
    if (diffval.tv_sec>=0) {
        if (diffval.tv_sec>0) {
            unsigned int tmp;
        
            tmp = diffval.tv_sec/refTick->tempo;
            refTick->curTick += tmp*1000000;
        
            tmp = diffval.tv_sec%refTick->tempo;
            tmp *= 1000000;
            refTick->curTick += tmp/refTick->tempo;
            diffval.tv_usec += tmp%refTick->tempo;
        }
        refTick->curTick += diffval.tv_usec/refTick->tempo;
        refTick->lastGetTimeOfDay = curval;
        refTick->lastGetTimeOfDay.tv_usec -= diffval.tv_usec%refTick->tempo;
        normalize_timeval(&refTick->lastGetTimeOfDay);
    }

    return refTick->curTick;
}


#if 0
int
main()
{
    tick_t tick;
    int next = 48;

    init_tick(96, &tick);
    for (;;) {
        update_tick(&tick);
        if (tick.curTick>next) {
            next += 48;
            printf("a\n");
        }
    }

    return 0;
}
#endif


/*******************************************************************
  get SMF file to buffer
 */

Bool /* isError */
get_smf_to_buffer(FILE *fp,
                  smf_buffer_t *refSMFBuffer /* result */)
{
    Bool result = False;
    buffer_t curBuf;
    int remain;

    {
        /* get temporary */
        size_t lenAdditional = INITIAL_BUFFER_SIZE/2;
        size_t lenRealRead;
        size_t curPos = 0;
        buffer_t buffer = NULL;

        do {
            lenAdditional *= 2;
            buffer = realloc(buffer, curPos+lenAdditional);
            curBuf = buffer+curPos;
            lenRealRead = fread(curBuf, 1, lenAdditional, fp);
            if (ferror(fp)) {
                fprintf(stderr, "read error.\n");
                result = True;
                goto quit;
            }
            curPos += lenRealRead;
            curBuf += lenRealRead;
        } while (lenAdditional == lenRealRead && !feof(fp));
        refSMFBuffer->pool = buffer;
        refSMFBuffer->lenPool = remain = curPos;
    }
    {
        /* get header */
        curBuf = refSMFBuffer->pool;
        if (get_header_chunk((cbuffer_t *)&curBuf, &refSMFBuffer->header)) {
            fprintf(stderr, "illegal header.\n");
            result = True;
            goto quit;
        }
        remain -=
            refSMFBuffer->header.length+LEN_MAGIC_HEADER_CHUNK
            + sizeof(dword_t);
        if (remain < refSMFBuffer->header.length) {
            fprintf(stderr, "too short file.\n");
            result = True;
            goto quit;
        }
        if (1 < refSMFBuffer->header.format) {
            fprintf(stderr, "format %i do not support.\n",
                    refSMFBuffer->header.format);
            result = True;
            goto quit;
        }
        if (ETUQPM != refSMFBuffer->header.timeUnitType) {
            fprintf(stderr, "time unit type HHMMSSFF do not support.\n");
            result = True;
            goto quit;
        }
    }
    {
        /* alloc track infos and fill infos */
        int i;

        refSMFBuffer->tracks =
            malloc(sizeof(track_info_t)*refSMFBuffer->header.numTracks);

        for (i=0; i<refSMFBuffer->header.numTracks; i++) {
            track_info_t *refCurTrack = &refSMFBuffer->tracks[i];

            if (!refSMFBuffer->flQuiet)
                fprintf(stderr, "Getting Track %i\n", i+1);

            refCurTrack->prevEvent = 0xf4;
            refCurTrack->portNumber = 1;
            if (get_track_chunk((cbuffer_t *)&curBuf, refCurTrack)) {
                fprintf(stderr, "illegal track chunk.\n");
                result = True;
                goto quit;
            }
            remain -= LEN_MAGIC_TRACK_CHUNK+sizeof(dword_t);
            if (remain < refCurTrack->length) {
                fprintf(stderr, "too short file.\n");
                result = True;
                goto quit;
            }
            refCurTrack->buffer = curBuf;
            curBuf += refCurTrack->length;
            remain -= refCurTrack->length;
        }
    }

    if (!refSMFBuffer->flQuiet) {
        /* display info */
        int i;
        
        fprintf(stderr,
                "total length = %ld\n", refSMFBuffer->header.length);
        fprintf(stderr,
                "format = %d\n", refSMFBuffer->header.format);
        fprintf(stderr,
                "number of tracks = %d\n", refSMFBuffer->header.numTracks);
        fprintf(stderr,
                "time unit = %d(%d)\n",
                refSMFBuffer->header.timeUnitValue,
                refSMFBuffer->header.timeUnitType);
        for (i=0; i<refSMFBuffer->header.numTracks; i++) {
            track_info_t *refCurTrack = &refSMFBuffer->tracks[i];
            fprintf(stderr, "Track %d: ", i+1);
            fprintf(stderr, "length = %ldbytes\n", refCurTrack->length);
        }
    }
quit:
    return result;
}


#if 0
int
main()
{
    static smf_buffer_t smfbuf;
    if (get_smf_to_buffer(stdin, &smfbuf)) {
        exit(1);
    }
    {
        int i;
        for (i=0; i<smfbuf.header.numTracks; i++) {
            track_info_t *refCurTrack = &smfbuf.tracks[i];
            printf("Track %i:\n", i+1);
            while (refCurTrack->remain>0) {
                event_t event;
                int j;
                {
                    int count;
                    int delta =
                        get_var_length_value(&refCurTrack->buffer, &count);
                    if (refCurTrack->remain<count) {
                        exit(1);
                    }
                    refCurTrack->remain -= count;
                    printf("  +%d(%d) : ", delta, count);
                }
                if (get_event(refCurTrack, &event)) {
                    if (refCurTrack->remain>0) {
                        exit(1);
                    }
                }
                printf("%02x", event.eventCode);
                for (j=0; j<event.lenParameter; j++) {
                    printf(" %02x", event.cRefParameters[j]);
                }
                printf("\n");
            }
        }
    }
}
#endif


/***************************************************************
  play SMF buffer
 */

void
play_smf_buffer(smf_buffer_t *refSMFBuffer,
                dword_t startBar,
                dword_t startQuarter,
                dword_t startTick)
{
    Bool isSkip = True;
    dword_t curTick;
    event_queue_t queue;
    dword_t nearestAriseTick;

    {
        curTick = startTick +=
            refSMFBuffer->header.timeUnitValue*(startQuarter+startBar*4);
    }

    queue_construct(&queue);

    {
        /* unshift top events of each track */
        int tr;
        for   (tr=0; tr<refSMFBuffer->header.numTracks; tr++) {
            track_info_t *refTrack = &refSMFBuffer->tracks[tr];
            event_t event;
            dword_t deltaTime;
            
            get_valid_event(refSMFBuffer, refTrack, &event, &deltaTime);
            queue_unshift(&queue, deltaTime, &event, refTrack);
        }
        if   (queue_peek_top_arise_tick(&queue,
                                        &nearestAriseTick))
            goto quit;
    }
    tick_init(&refSMFBuffer->tick,
              refSMFBuffer->header.timeUnitValue,
              startTick);

    while (1) {
        if   (isSkip) {
            if (startTick<=nearestAriseTick) {
                isSkip = False;
                tick_start(&refSMFBuffer->tick);
            }
        } else
            curTick = tick_update(&refSMFBuffer->tick);
        if  (curTick>nearestAriseTick) {
            /* process timeout event */
            track_info_t *refTrack;
            event_t event;
            if (!queue_shift(&queue, &event, &refTrack)) {
                {
                    /* process */
                    switch (event.eventCode & 0xf0) {
                    case 0x80:
                    case 0x90:
                        if (isSkip)
                            goto next;
                        break;
                    case 0xf0:
                        switch (event.eventCode & 0x0f) {
                        case 0x05:
                            /* change port for current track */
                            refTrack->portNumber = *event.cRefParameters;
                            goto next;
                        case 0x0f:
                            switch  (*event.cRefParameters) {
                            case 0x21:
                                /* change port for current track */
                                refTrack->portNumber = event.cRefParameters[2];
                            goto next;
                            case 0x2f:
                                /* end of track */
                                goto eot;
                            case 0x51:
                                /* change tempo */
                                refSMFBuffer->tick.tempo = 
                                    ((dword_t)(event.cRefParameters[2]<<16)+
                                     (dword_t)(event.cRefParameters[3]<<8)+
                                     (dword_t)event.cRefParameters[4])
                                    /refSMFBuffer->header.timeUnitValue;
                                goto next;
                            }
                        }
                    }
                    /* change port */
                    if   (refSMFBuffer->portNumber != refTrack->portNumber) {
                        put_midi_message(0xf5);
                        put_midi_message(refTrack->portNumber);
                        refSMFBuffer->portNumber = refTrack->portNumber;
                    }
                    /* put MIDI event */
                    put_midi_message(event.eventCode);
                    {
                        int i;
                        for (i=0; i<event.lenParameter; i++)
                            put_midi_message(event.cRefParameters[i]);
                    }
                }
            next:
                {
                    /* unshift next event */
                    dword_t deltaTime;
                    get_valid_event(refSMFBuffer, refTrack,
                                    &event, &deltaTime);
                    queue_unshift(&queue,
                                  deltaTime+nearestAriseTick,
                                  &event, refTrack);
                }
            } else
                /* end of track
                   - explicit decrement for number of element in queue */
                ;
        eot:
            if (queue_peek_top_arise_tick(&queue,
                                          &nearestAriseTick)) {
                break;
            }
        } else
            /* sleep */
            usleep((nearestAriseTick-curTick)*
                   refSMFBuffer->tick.tempo);
    }

quit:
    queue_destruct(&queue);
}


/******************************************************************
  main rootine
 */

/* signal handling */

void
sig_handler(int dum)
{
    (void)dum;
    reset_all_midi();
    exit(1);
}


/* usage */

void
usage(void)
{
    fprintf(stderr, "usage: smfplay [-s <n>] <filename>\n");
    fprintf(stderr, "  -s : skip n ticks.\n");
    fprintf(stderr, "  -q : quiet.\n");
}

/* main */
int
main(int ac, char *av[])
{
    int result = 0;
    dword_t skipBar = 0;
    dword_t skipQuarter = 0;
    dword_t skipTick = 0;
    smf_buffer_t smfBuf;
    char const *fileName = "-";
    FILE *fp = NULL;

#ifndef NOSETPRIO
    struct rtprio rtp;
    rtp.type = RTP_PRIO_REALTIME;
    rtp.prio = 0;
    /*    setpriority(PRIO_PROCESS, getpid(), -20);*/
    if (rtprio(RTP_SET, getpid(), &rtp)) {
        perror("rtprio");
    }
#endif
    memset((void *)&smfBuf, 0, sizeof(smfBuf));

    attach_midi_port();

    reset_all_midi();
    
    if (1 == ac) {
        usage();
        result = 1;
        goto quit;
    }
    {
        int i;
        for (i=1; i<ac; i++) {
            if (!strcmp(av[i], "-s")) {
                char *p = av[i+1];

                i++;
                skipBar = strtoul(p, &p, 10);
                if (':' != *p)
                    goto exit_opt_s;
                skipQuarter = strtoul(p+1, &p, 10);
                if (':' != *p)
                    goto exit_opt_s;
                skipTick = strtoul(p+1, &p, 10);
            exit_opt_s:
                continue;
            }
            if (!strcmp(av[i], "-q")) {
                smfBuf.flQuiet = True;
                continue;
            }
            if (i!=(ac-1)) {
                usage();
                result = 1;
                goto quit;
            }
            fileName = av[i];
        }
    }

    if (!strcmp(fileName, "-")) {
        fp = stdin;
    } else
        if (NULL == (fp = fopen(fileName, "r"))) {
            perror("fopen");
            result = 1;
            goto quit;
        }
    if (get_smf_to_buffer(fp, &smfBuf)) {
        result = 1;
        goto quit;
    }
    signal(SIGHUP, sig_handler);
    signal(SIGINT, sig_handler);
    signal(SIGTERM, sig_handler);
    play_smf_buffer(&smfBuf, skipBar, skipQuarter, skipTick);
    
quit:
    return result;
}


/* end of file */
