#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <sys/time.h>
#include <signal.h>

typedef unsigned char byte_t;
typedef byte_t *buffer_t;
typedef byte_t const *cbuffer_t;
typedef unsigned short word_t;
typedef unsigned long dword_t;
#ifndef Bool
#define Bool int
#define True (!0)
#define False (0)
#endif

/* tick generator */
typedef struct
{
    long tempo;    /* microseconds for an unit of timebase */
    dword_t curTick;
    struct timeval lastGetTimeOfDay;
} tick_t;

static void
normalize_timeval(struct timeval *refTimeval /* I/O */)
{
    while (refTimeval->tv_usec >= 1000000) {
        refTimeval->tv_sec++;
        refTimeval->tv_usec -= 1000000;
    }
    while (refTimeval->tv_usec < 0) {
        refTimeval->tv_sec--;
        refTimeval->tv_usec += 1000000;
    }
}

/* init tick */
void
tick_init(tick_t *refTick /* object */,
          unsigned int timebase, dword_t initialTick)
{
    /*
      120 qpm
      ---  1000000*60/(timebase/120) = 500000/timebase
     */
    refTick->tempo = 500000U/timebase;
    refTick->curTick = 0;
    gettimeofday(&refTick->lastGetTimeOfDay, NULL);
    {
        /* take margin */
        struct timeval tmp;
        tmp.tv_sec = 0;
        tmp.tv_usec = 50000;
        normalize_timeval(&tmp);
        timeradd(&tmp, &refTick->lastGetTimeOfDay, &refTick->lastGetTimeOfDay);
    }
}

/* update tick */
dword_t /* current tick */
tick_update(tick_t *refTick /* object */)
{
    struct timeval curval, diffval;

    gettimeofday(&curval, NULL);
    timersub(&curval, &refTick->lastGetTimeOfDay, &diffval);
    if (diffval.tv_sec>=0) {
        if (diffval.tv_sec>0) {
            unsigned int tmp;
        
            tmp = diffval.tv_sec/refTick->tempo;
            refTick->curTick += tmp*1000000;
        
            tmp = diffval.tv_sec%refTick->tempo;
            tmp *= 1000000;
            refTick->curTick += tmp/refTick->tempo;
            diffval.tv_usec += tmp%refTick->tempo;
        }
        refTick->curTick += diffval.tv_usec/refTick->tempo;
        refTick->lastGetTimeOfDay = curval;
        refTick->lastGetTimeOfDay.tv_usec -= diffval.tv_usec%refTick->tempo;
        normalize_timeval(&refTick->lastGetTimeOfDay);
    }

    return refTick->curTick;
}


int
main()
{
    tick_t t;
    dword_t current = 0;
    dword_t dum = 0;

    tick_init(&t, 48, 0);
    
    while (1) {
        dword_t tmp;
        for (dum=0; dum<1000000; dum++)
            ;
        tmp = tick_update(&t);
        if (current != tmp) {
            printf("%d ", current=tmp);
            fflush(stdout);
        }
    }
    return 0;
}
